/**
 * 稳定的机器码获取模块
 * 
 * 核心思路：
 * 1. 优先使用物理网卡MAC地址（最稳定的硬件标识）
 * 2. 结合主板序列号和CPU信息作为补充
 * 3. 将获取到的标识缓存到本地，确保一致性
 * 4. 即使 cursor-free-vip 重置机器码，也能保持稳定
 */

const { exec } = require('child_process');
const crypto = require('crypto');
const os = require('os');
const fs = require('fs');
const path = require('path');

// 缓存文件路径（存储在用户数据目录）
const CACHE_DIR = path.join(process.env.APPDATA || os.homedir(), '.cursor-renewal-cache');
const CACHE_FILE = path.join(CACHE_DIR, 'stable-machine-id.json');

/**
 * 执行 PowerShell 命令
 */
function execPowerShell(command) {
  return new Promise((resolve, reject) => {
    exec(`powershell -Command "${command}"`, { timeout: 10000 }, (error, stdout, stderr) => {
      if (error) {
        reject(error);
        return;
      }
      resolve(stdout.trim());
    });
  });
}

/**
 * 执行普通命令
 */
function execCommand(command) {
  return new Promise((resolve, reject) => {
    exec(command, { timeout: 10000 }, (error, stdout, stderr) => {
      if (error) {
        reject(error);
        return;
      }
      resolve(stdout.trim());
    });
  });
}

/**
 * 获取物理网卡的MAC地址（排除虚拟网卡）
 * 这是最稳定的硬件标识，不会因为重置机器码而改变
 */
async function getPhysicalMacAddress() {
  try {
    // 方法1: 使用 getmac 命令（最可靠）
    try {
      const result = await execCommand('getmac /fo csv /nh');
      const lines = result.split('\n').filter(line => line.trim());
      
      // 过滤掉虚拟网卡和无效MAC
      const validMacs = [];
      for (const line of lines) {
        const match = line.match(/([0-9A-F]{2}-[0-9A-F]{2}-[0-9A-F]{2}-[0-9A-F]{2}-[0-9A-F]{2}-[0-9A-F]{2})/i);
        if (match) {
          const mac = match[1].replace(/-/g, ':').toUpperCase();
          
          // 排除明显的虚拟MAC地址前缀
          const virtualPrefixes = [
            '00:05:69', // VMware
            '00:0C:29', // VMware
            '00:50:56', // VMware
            '00:1C:14', // VMware
            '00:1C:42', // Parallels
            '00:15:5D', // Hyper-V
            '08:00:27', // VirtualBox
            '0A:00:27', // VirtualBox
            '00:00:00:00:00:00', // Invalid
          ];
          
          const isVirtual = virtualPrefixes.some(prefix => mac.startsWith(prefix));
          if (!isVirtual && mac !== '00:00:00:00:00:00') {
            validMacs.push(mac);
          }
        }
      }
      
      if (validMacs.length > 0) {
        // 返回第一个有效的物理MAC地址
        console.log(`✅ 获取到物理MAC地址 (getmac): ${validMacs[0]}`);
        return validMacs[0];
      }
    } catch (e) {
      console.warn('getmac 命令失败:', e.message);
    }

    // 方法2: 使用 WMI 查询网卡信息（包含更详细的过滤）
    try {
      const wmiQuery = `Get-WmiObject Win32_NetworkAdapter | Where-Object {$_.PhysicalAdapter -eq $true -and $_.MACAddress -ne $null -and $_.AdapterTypeID -eq 0} | Select-Object -First 1 -ExpandProperty MACAddress`;
      const mac = await execPowerShell(wmiQuery);
      
      if (mac && mac !== '00:00:00:00:00:00') {
        const formattedMac = mac.replace(/:/g, ':').toUpperCase();
        console.log(`✅ 获取到物理MAC地址 (WMI): ${formattedMac}`);
        return formattedMac;
      }
    } catch (e) {
      console.warn('WMI 查询失败:', e.message);
    }

    // 方法3: 使用 Node.js os.networkInterfaces()（最后的备选）
    const interfaces = os.networkInterfaces();
    for (const name of Object.keys(interfaces)) {
      // 排除虚拟网卡名称
      const virtualNames = ['vmware', 'virtualbox', 'vethernet', 'hyper-v', 'vboxnet', 'docker', 'wsl'];
      if (virtualNames.some(vn => name.toLowerCase().includes(vn))) {
        continue;
      }

      for (const net of interfaces[name]) {
        if (!net.internal && net.mac && net.mac !== '00:00:00:00:00:00') {
          const mac = net.mac.toUpperCase();
          // 再次检查虚拟MAC前缀
          const virtualPrefixes = ['00:05:69', '00:0C:29', '00:50:56', '00:1C:14', '00:1C:42', '00:15:5D', '08:00:27', '0A:00:27'];
          const isVirtual = virtualPrefixes.some(prefix => mac.startsWith(prefix));
          
          if (!isVirtual) {
            console.log(`✅ 获取到物理MAC地址 (os.networkInterfaces): ${mac}`);
            return mac;
          }
        }
      }
    }

    throw new Error('未找到有效的物理网卡MAC地址');
  } catch (error) {
    console.error('获取物理MAC地址失败:', error.message);
    throw error;
  }
}

/**
 * 获取主板序列号（作为辅助标识）
 */
async function getMotherboardSerial() {
  try {
    const result = await execCommand('wmic baseboard get serialnumber');
    const lines = result.split('\n').filter(line => line.trim() && !line.includes('SerialNumber'));
    const serial = lines[0]?.trim();
    
    if (serial && serial !== 'Default string' && serial !== 'To be filled by O.E.M.') {
      console.log(`✅ 获取到主板序列号: ${serial.substring(0, 8)}...`);
      return serial;
    }
  } catch (error) {
    console.warn('获取主板序列号失败:', error.message);
  }
  return null;
}

/**
 * 获取CPU ID（作为辅助标识）
 */
async function getCpuId() {
  try {
    const result = await execCommand('wmic cpu get processorid');
    const lines = result.split('\n').filter(line => line.trim() && !line.includes('ProcessorId'));
    const cpuId = lines[0]?.trim();
    
    if (cpuId) {
      console.log(`✅ 获取到CPU ID: ${cpuId.substring(0, 8)}...`);
      return cpuId;
    }
  } catch (error) {
    console.warn('获取CPU ID失败:', error.message);
  }
  return null;
}

/**
 * 获取硬盘序列号（作为辅助标识）
 */
async function getDiskSerial() {
  try {
    const result = await execCommand('wmic diskdrive get serialnumber');
    const lines = result.split('\n').filter(line => line.trim() && !line.includes('SerialNumber'));
    const serial = lines[0]?.trim();
    
    if (serial) {
      console.log(`✅ 获取到硬盘序列号: ${serial.substring(0, 8)}...`);
      return serial;
    }
  } catch (error) {
    console.warn('获取硬盘序列号失败:', error.message);
  }
  return null;
}

/**
 * 生成稳定的机器码
 * 组合多个硬件标识，确保即使某个标识改变，整体仍然稳定
 */
async function generateStableMachineId() {
  console.log('\n🔍 开始获取稳定的硬件标识...\n');

  // 1. 获取物理MAC地址（最重要，必须成功）
  const macAddress = await getPhysicalMacAddress();
  
  // 2. 获取其他辅助标识（失败不影响整体）
  const motherboardSerial = await getMotherboardSerial();
  const cpuId = await getCpuId();
  const diskSerial = await getDiskSerial();
  
  // 3. 组合所有标识
  const components = [
    macAddress,  // 物理MAC地址（权重最高）
    motherboardSerial || '',
    cpuId || '',
    diskSerial || ''
  ].filter(c => c);

  // 4. 生成唯一标识（使用SHA-256）
  const combined = components.join('|');
  const hash = crypto.createHash('sha256').update(combined).digest('hex');
  
  console.log('\n✅ 硬件标识组合完成:');
  console.log(`   - 物理MAC地址: ${macAddress}`);
  console.log(`   - 主板序列号: ${motherboardSerial || '未获取'}`);
  console.log(`   - CPU ID: ${cpuId || '未获取'}`);
  console.log(`   - 硬盘序列号: ${diskSerial || '未获取'}`);
  console.log(`   - 最终机器码: ${hash}\n`);

  return {
    machineId: hash,
    components: {
      macAddress,
      motherboardSerial,
      cpuId,
      diskSerial
    },
    timestamp: new Date().toISOString()
  };
}

/**
 * 从缓存读取机器码
 */
function readCache() {
  try {
    if (fs.existsSync(CACHE_FILE)) {
      const data = fs.readFileSync(CACHE_FILE, 'utf8');
      return JSON.parse(data);
    }
  } catch (error) {
    console.warn('读取缓存失败:', error.message);
  }
  return null;
}

/**
 * 写入缓存
 */
function writeCache(data) {
  try {
    // 确保目录存在
    if (!fs.existsSync(CACHE_DIR)) {
      fs.mkdirSync(CACHE_DIR, { recursive: true });
    }
    
    fs.writeFileSync(CACHE_FILE, JSON.stringify(data, null, 2), 'utf8');
    console.log(`✅ 机器码已缓存到: ${CACHE_FILE}`);
  } catch (error) {
    console.error('写入缓存失败:', error.message);
  }
}

/**
 * 验证缓存的机器码是否仍然有效
 * 主要检查物理MAC地址是否匹配
 */
async function verifyCachedMachineId(cached) {
  try {
    const currentMac = await getPhysicalMacAddress();
    
    // 如果物理MAC地址匹配，认为缓存有效
    if (cached.components && cached.components.macAddress === currentMac) {
      console.log('✅ 缓存的机器码验证通过（MAC地址匹配）');
      return true;
    }
    
    console.log('⚠️  缓存的机器码验证失败（MAC地址不匹配）');
    return false;
  } catch (error) {
    console.warn('验证缓存失败:', error.message);
    return false;
  }
}

/**
 * 获取稳定的机器码（主函数）
 * 
 * 逻辑：
 * 1. 先尝试从缓存读取
 * 2. 如果缓存存在且有效，直接使用
 * 3. 否则重新生成并缓存
 */
async function getStableMachineId(forceRegenerate = false) {
  try {
    // 如果不强制重新生成，先尝试使用缓存
    if (!forceRegenerate) {
      const cached = readCache();
      if (cached && cached.machineId) {
        console.log('📦 发现缓存的机器码，正在验证...');
        
        const isValid = await verifyCachedMachineId(cached);
        if (isValid) {
          console.log('✅ 使用缓存的机器码:', cached.machineId);
          return cached.machineId;
        }
      }
    }

    // 生成新的机器码
    console.log(forceRegenerate ? '🔄 强制重新生成机器码...' : '🆕 生成新的机器码...');
    const result = await generateStableMachineId();
    
    // 缓存到本地
    writeCache(result);
    
    return result.machineId;
  } catch (error) {
    console.error('❌ 获取稳定机器码失败:', error.message);
    
    // 如果所有方法都失败，尝试返回缓存（即使可能无效）
    const cached = readCache();
    if (cached && cached.machineId) {
      console.warn('⚠️  使用缓存的机器码作为后备方案');
      return cached.machineId;
    }
    
    throw error;
  }
}

/**
 * 清除缓存（用于测试或重置）
 */
function clearCache() {
  try {
    if (fs.existsSync(CACHE_FILE)) {
      fs.unlinkSync(CACHE_FILE);
      console.log('✅ 机器码缓存已清除');
      return true;
    }
    console.log('ℹ️  缓存文件不存在');
    return false;
  } catch (error) {
    console.error('❌ 清除缓存失败:', error.message);
    return false;
  }
}

module.exports = {
  getStableMachineId,
  generateStableMachineId,
  getPhysicalMacAddress,
  clearCache,
  getCacheFilePath: () => CACHE_FILE
};

// 如果直接运行此脚本，执行测试
if (require.main === module) {
  (async () => {
    console.log('='.repeat(70));
    console.log('稳定机器码获取模块 - 测试');
    console.log('='.repeat(70));
    
    try {
      const machineId = await getStableMachineId();
      console.log('\n' + '='.repeat(70));
      console.log('最终机器码（SHA-256）:', machineId);
      console.log('='.repeat(70));
      console.log('\n✅ 此机器码基于物理硬件，即使重置 Cursor 机器码也不会改变！\n');
    } catch (error) {
      console.error('\n❌ 测试失败:', error.message);
      process.exit(1);
    }
  })();
}

