#!/usr/bin/env node

/**
 * 新旧机器码对比工具
 * 
 * 用途：
 * 1. 同时显示新旧两种方法获取的机器码
 * 2. 便于理解为什么旧方法会受重置影响
 * 3. 验证新方法的稳定性
 */

const { execSync } = require('child_process');
const crypto = require('crypto');
const os = require('os');
const fs = require('fs');
const path = require('path');

console.log('='.repeat(80));
console.log('新旧机器码对比工具');
console.log('='.repeat(80));
console.log('');

// ========================================
// 方法1: 旧方法 (node-machine-id)
// ========================================
console.log('【方法 1】旧方法 - 使用 node-machine-id');
console.log('-'.repeat(80));

let oldMethod = {
  raw: null,
  hashed: null,
  source: null,
  stable: false
};

try {
  const { machineIdSync } = require('node-machine-id');
  oldMethod.raw = machineIdSync({ original: true });
  oldMethod.hashed = crypto.createHash('sha256').update(oldMethod.raw).digest('hex');
  oldMethod.source = 'Windows 注册表 (HKEY_LOCAL_MACHINE\\SOFTWARE\\Microsoft\\Cryptography\\MachineGuid)';
  oldMethod.stable = false;
  
  console.log(`✅ 原始值: ${oldMethod.raw}`);
  console.log(`✅ SHA-256: ${oldMethod.hashed}`);
  console.log(`📍 数据来源: ${oldMethod.source}`);
  console.log(`⚠️  稳定性: 不稳定（会被 cursor-free-vip 重置）`);
} catch (error) {
  console.log(`❌ 获取失败: ${error.message}`);
}

console.log('');

// ========================================
// 方法2: 新方法 (物理MAC地址)
// ========================================
console.log('【方法 2】新方法 - 使用物理MAC地址');
console.log('-'.repeat(80));

let newMethod = {
  macAddress: null,
  motherboardSerial: null,
  cpuId: null,
  combined: null,
  hashed: null,
  stable: true
};

try {
  // 获取物理MAC地址
  const result = execSync('getmac /fo csv /nh', { encoding: 'utf8', timeout: 5000 });
  const lines = result.split('\n').filter(line => line.trim());
  
  for (const line of lines) {
    const match = line.match(/([0-9A-F]{2}-[0-9A-F]{2}-[0-9A-F]{2}-[0-9A-F]{2}-[0-9A-F]{2}-[0-9A-F]{2})/i);
    if (match) {
      const mac = match[1].replace(/-/g, ':').toUpperCase();
      const virtualPrefixes = ['00:05:69', '00:0C:29', '00:50:56', '00:1C:14', '00:1C:42', '00:15:5D', '08:00:27', '0A:00:27'];
      const isVirtual = virtualPrefixes.some(prefix => mac.startsWith(prefix));
      
      if (!isVirtual && mac !== '00:00:00:00:00:00') {
        newMethod.macAddress = mac;
        break;
      }
    }
  }
  
  // 获取辅助信息
  try {
    const mbResult = execSync('wmic baseboard get serialnumber', { encoding: 'utf8', timeout: 3000 });
    const mbLines = mbResult.split('\n').filter(line => line.trim() && !line.includes('SerialNumber'));
    newMethod.motherboardSerial = mbLines[0]?.trim() || '';
    if (newMethod.motherboardSerial === 'Default string' || newMethod.motherboardSerial === 'To be filled by O.E.M.') {
      newMethod.motherboardSerial = '';
    }
  } catch (e) {}
  
  try {
    const cpuResult = execSync('wmic cpu get processorid', { encoding: 'utf8', timeout: 3000 });
    const cpuLines = cpuResult.split('\n').filter(line => line.trim() && !line.includes('ProcessorId'));
    newMethod.cpuId = cpuLines[0]?.trim() || '';
  } catch (e) {}
  
  // 生成机器码
  const components = [newMethod.macAddress, newMethod.motherboardSerial, newMethod.cpuId].filter(c => c);
  newMethod.combined = components.join('|');
  newMethod.hashed = crypto.createHash('sha256').update(newMethod.combined).digest('hex');
  
  console.log(`✅ 物理MAC地址: ${newMethod.macAddress} (主要标识)`);
  if (newMethod.motherboardSerial) {
    console.log(`✅ 主板序列号: ${newMethod.motherboardSerial} (辅助)`);
  }
  if (newMethod.cpuId) {
    console.log(`✅ CPU ID: ${newMethod.cpuId} (辅助)`);
  }
  console.log(`✅ 组合字符串: ${newMethod.combined}`);
  console.log(`✅ SHA-256: ${newMethod.hashed}`);
  console.log(`📍 数据来源: 物理硬件（网卡芯片）`);
  console.log(`✅ 稳定性: 非常稳定（不受软件重置影响）`);
} catch (error) {
  console.log(`❌ 获取失败: ${error.message}`);
}

console.log('');

// ========================================
// 对比分析
// ========================================
console.log('【对比分析】');
console.log('='.repeat(80));
console.log('');

console.log('┌─────────────────┬─────────────────────────────────────────────────────┐');
console.log('│ 对比项          │ 说明                                                │');
console.log('├─────────────────┼─────────────────────────────────────────────────────┤');
console.log('│ 数据来源        │ 旧: 注册表MachineGuid (软件级)                      │');
console.log('│                 │ 新: 物理网卡MAC地址 (硬件级)                        │');
console.log('├─────────────────┼─────────────────────────────────────────────────────┤');
console.log('│ 稳定性          │ 旧: ❌ 不稳定                                       │');
console.log('│                 │ 新: ✅ 非常稳定                                     │');
console.log('├─────────────────┼─────────────────────────────────────────────────────┤');
console.log('│ 受重置影响      │ 旧: ❌ 会改变                                       │');
console.log('│                 │ 新: ✅ 不会改变                                     │');
console.log('├─────────────────┼─────────────────────────────────────────────────────┤');
console.log('│ 硬件更换        │ 旧: 可能变（取决于注册表）                          │');
console.log('│                 │ 新: 会变（更换网卡/主板）                           │');
console.log('├─────────────────┼─────────────────────────────────────────────────────┤');
console.log('│ 重装系统        │ 旧: 可能变                                          │');
console.log('│                 │ 新: 不变（硬件未变）                                │');
console.log('├─────────────────┼─────────────────────────────────────────────────────┤');
console.log('│ 管理员权限      │ 旧: 建议需要                                        │');
console.log('│                 │ 新: 不需要（MAC地址获取）                           │');
console.log('└─────────────────┴─────────────────────────────────────────────────────┘');
console.log('');

// ========================================
// 机器码对比
// ========================================
if (oldMethod.hashed && newMethod.hashed) {
  console.log('【机器码对比】');
  console.log('-'.repeat(80));
  console.log('');
  console.log(`旧方法机器码: ${oldMethod.hashed}`);
  console.log(`新方法机器码: ${newMethod.hashed}`);
  console.log('');
  
  if (oldMethod.hashed === newMethod.hashed) {
    console.log('✅ 两种方法的机器码一致');
    console.log('   （当前环境下恰好相同，但使用重置功能后会不同）');
  } else {
    console.log('⚠️  两种方法的机器码不同');
    console.log('   这是正常现象，因为数据来源不同');
    console.log('');
    console.log('   重要说明：');
    console.log('   • 应用程序现在使用新方法（基于物理MAC地址）');
    console.log('   • 新方法的机器码不会因 cursor-free-vip 重置而改变');
    console.log('   • 旧方法仅用于对比，不再使用');
  }
  console.log('');
}

// ========================================
// 重置影响测试说明
// ========================================
console.log('【重置影响测试】');
console.log('='.repeat(80));
console.log('');
console.log('要验证新方法是否真的稳定，请按以下步骤操作：');
console.log('');
console.log('步骤 1: 记录当前机器码');
console.log('-------');
console.log(`  新方法机器码: ${newMethod.hashed || 'N/A'}`);
console.log('');
console.log('步骤 2: 使用 cursor-free-vip 重置 Cursor 机器码');
console.log('-------');
console.log('  cd cursor-free-vip-main');
console.log('  python main.py');
console.log('  → 选择"重置机器码"功能');
console.log('');
console.log('步骤 3: 再次运行此脚本');
console.log('-------');
console.log('  node 对比新旧机器码.js');
console.log('');
console.log('步骤 4: 对比结果');
console.log('-------');
console.log('  ✅ 新方法机器码应该保持不变');
console.log('  ⚠️  旧方法机器码会改变');
console.log('');
console.log('如果新方法机器码也改变了，说明：');
console.log('  1. 可能读取了注册表（说明 getmac 命令失败）');
console.log('  2. 网卡配置发生了实际变化');
console.log('  → 请检查网络适配器设置');
console.log('');

// ========================================
// 缓存信息
// ========================================
const CACHE_DIR = path.join(process.env.APPDATA || os.homedir(), '.cursor-renewal-cache');
const CACHE_FILE = path.join(CACHE_DIR, 'stable-machine-id.json');

console.log('【缓存信息】');
console.log('='.repeat(80));
console.log('');
console.log(`缓存文件: ${CACHE_FILE}`);
console.log('');

if (fs.existsSync(CACHE_FILE)) {
  try {
    const cachedData = JSON.parse(fs.readFileSync(CACHE_FILE, 'utf8'));
    console.log('缓存内容:');
    console.log(JSON.stringify(cachedData, null, 2));
    console.log('');
    
    if (cachedData.machineId === newMethod.hashed) {
      console.log('✅ 缓存的机器码与当前新方法机器码一致');
    } else {
      console.log('⚠️  缓存的机器码与当前新方法机器码不一致');
      console.log('   建议清除缓存重新生成：');
      console.log(`   del "${CACHE_FILE}"`);
    }
  } catch (e) {
    console.log(`❌ 读取缓存失败: ${e.message}`);
  }
} else {
  console.log('ℹ️  缓存文件不存在（首次运行应用时会自动创建）');
}

console.log('');
console.log('='.repeat(80));
console.log('');

