#!/usr/bin/env node

/**
 * 模拟应用程序中的机器码获取逻辑
 * 与 main.js 中的 generateMachineId() 函数完全一致
 */

const crypto = require('crypto');
const { v4: uuidv4 } = require('uuid');

// 生成设备ID（与main.js第3631-3648行完全一致）
function generateMachineId() {
  try {
    const { machineIdSync } = require('node-machine-id');
    // Windows系统必须使用 {original: true} 参数以获取稳定的机器码
    // 否则每次获取的机器码可能不同，导致"key已绑定其他机器码"错误
    const rawMachineId = machineIdSync({ original: true });
    
    // 对原始机器码进行SHA-256加密，与后端存储格式保持一致
    const hashedMachineId = crypto.createHash('sha256').update(rawMachineId).digest('hex');
    
    return hashedMachineId;
  } catch (error) {
    console.warn('无法获取真实的机器ID，使用随机ID:', error.message);
    // 对随机ID也进行SHA-256加密
    return crypto.createHash('sha256').update(uuidv4()).digest('hex');
  }
}

// 获取原始机器码（用于调试）
function getRawMachineId() {
  try {
    const { machineIdSync } = require('node-machine-id');
    return machineIdSync({ original: true });
  } catch (error) {
    return null;
  }
}

console.log('='.repeat(70));
console.log('🔍 模拟应用程序换号时的机器码获取');
console.log('='.repeat(70));
console.log('');

console.log('📋 当前环境:');
console.log(`  • 操作系统: ${process.platform}`);
console.log(`  • Node版本: ${process.version}`);
console.log('');

// 首先显示原始机器码和加密后的机器码
const rawId = getRawMachineId();
if (rawId) {
  console.log('🔐 机器码信息:');
  console.log(`  • 原始机器码: ${rawId}`);
  const hashed = crypto.createHash('sha256').update(rawId).digest('hex');
  console.log(`  • 加密后机器码: ${hashed}`);
  console.log(`  • 加密算法: SHA-256`);
  console.log(`  • 加密后长度: ${hashed.length} (64位十六进制)`);
  console.log('');
}

console.log('🔄 模拟换号操作（连续3次）...');
console.log('-'.repeat(70));

const machineIds = [];
for (let i = 1; i <= 3; i++) {
  console.log(`\n第${i}次换号:`);
  try {
    const machineId = generateMachineId();
    machineIds.push(machineId);
    console.log(`  ✅ 发送给后端的机器码: ${machineId}`);
    console.log(`  📏 机器码长度: ${machineId.length}`);
  } catch (error) {
    console.error(`  ❌ 获取失败: ${error.message}`);
  }
}

console.log('');
console.log('='.repeat(70));
console.log('📊 测试结果');
console.log('='.repeat(70));
console.log('');

// 检查一致性
const uniqueIds = [...new Set(machineIds)];

if (uniqueIds.length === 1) {
  console.log('✅ 机器码稳定！不会出现"key已绑定其他机器码"的问题');
  console.log('');
  console.log(`📱 发送给后端的机器码（SHA-256加密）: ${uniqueIds[0]}`);
  console.log('');
  console.log('💡 修复说明:');
  console.log('  • Windows系统使用 machineIdSync({ original: true }) 获取稳定的原始机器码');
  console.log('  • 对原始机器码进行 SHA-256 加密，生成64位十六进制字符串');
  console.log('  • 用户在同一台机器上多次换号，加密后的机器码保持一致');
  console.log('  • 不会出现"key已绑定其他机器码"的错误提示');
  console.log('');
  console.log('✅ 修复成功！');
} else {
  console.log('❌ 警告：机器码不稳定！');
  console.log('');
  console.log(`  发现 ${uniqueIds.length} 个不同的机器码:`);
  uniqueIds.forEach((id, idx) => {
    const count = machineIds.filter(m => m === id).length;
    console.log(`    ${idx + 1}. ${id} (出现${count}次)`);
  });
  console.log('');
  console.log('⚠️ 这会导致用户换号时提示"key已绑定其他机器码"');
}

console.log('');
console.log('='.repeat(70));
console.log('');

