#!/usr/bin/env node

/**
 * 测试稳定机器码获取模块
 * 
 * 用途：
 * 1. 测试物理MAC地址获取是否正常
 * 2. 对比新旧机器码获取方式
 * 3. 验证机器码在重置后是否保持稳定
 */

const { execSync } = require('child_process');
const crypto = require('crypto');
const os = require('os');
const fs = require('fs');
const path = require('path');

console.log('='.repeat(80));
console.log('稳定机器码测试工具');
console.log('='.repeat(80));
console.log('');

// ========================================
// 1. 测试物理MAC地址获取
// ========================================
console.log('【步骤 1】获取物理MAC地址\n');

let macAddress = null;

try {
  const result = execSync('getmac /fo csv /nh', { encoding: 'utf8', timeout: 5000 });
  const lines = result.split('\n').filter(line => line.trim());
  
  console.log('找到的所有网卡:');
  lines.forEach((line, index) => {
    const match = line.match(/([0-9A-F]{2}-[0-9A-F]{2}-[0-9A-F]{2}-[0-9A-F]{2}-[0-9A-F]{2}-[0-9A-F]{2})/i);
    if (match) {
      const mac = match[1].replace(/-/g, ':').toUpperCase();
      const virtualPrefixes = ['00:05:69', '00:0C:29', '00:50:56', '00:1C:14', '00:1C:42', '00:15:5D', '08:00:27', '0A:00:27'];
      const isVirtual = virtualPrefixes.some(prefix => mac.startsWith(prefix));
      
      console.log(`  ${index + 1}. ${mac} ${isVirtual ? '[虚拟网卡]' : '[物理网卡] ✅'}`);
      
      if (!isVirtual && mac !== '00:00:00:00:00:00' && !macAddress) {
        macAddress = mac;
      }
    }
  });
  
  console.log('');
  if (macAddress) {
    console.log(`✅ 选择的物理MAC地址: ${macAddress}\n`);
  } else {
    throw new Error('未找到有效的物理网卡');
  }
} catch (error) {
  console.error(`❌ 获取MAC地址失败: ${error.message}\n`);
  process.exit(1);
}

// ========================================
// 2. 获取其他硬件信息
// ========================================
console.log('【步骤 2】获取其他硬件信息\n');

let motherboardSerial = '';
let cpuId = '';
let diskSerial = '';

try {
  const mbResult = execSync('wmic baseboard get serialnumber', { encoding: 'utf8', timeout: 3000 });
  const mbLines = mbResult.split('\n').filter(line => line.trim() && !line.includes('SerialNumber'));
  motherboardSerial = mbLines[0]?.trim() || '';
  if (motherboardSerial === 'Default string' || motherboardSerial === 'To be filled by O.E.M.') {
    motherboardSerial = '';
  }
  if (motherboardSerial) {
    console.log(`✅ 主板序列号: ${motherboardSerial}`);
  } else {
    console.log('⚠️  主板序列号: 未获取到（部分主板不支持）');
  }
} catch (e) {
  console.log(`⚠️  主板序列号: 获取失败`);
}

try {
  const cpuResult = execSync('wmic cpu get processorid', { encoding: 'utf8', timeout: 3000 });
  const cpuLines = cpuResult.split('\n').filter(line => line.trim() && !line.includes('ProcessorId'));
  cpuId = cpuLines[0]?.trim() || '';
  if (cpuId) {
    console.log(`✅ CPU ID: ${cpuId}`);
  } else {
    console.log('⚠️  CPU ID: 未获取到');
  }
} catch (e) {
  console.log(`⚠️  CPU ID: 获取失败`);
}

try {
  const diskResult = execSync('wmic diskdrive get serialnumber', { encoding: 'utf8', timeout: 3000 });
  const diskLines = diskResult.split('\n').filter(line => line.trim() && !line.includes('SerialNumber'));
  diskSerial = diskLines[0]?.trim() || '';
  if (diskSerial) {
    console.log(`✅ 硬盘序列号: ${diskSerial}`);
  } else {
    console.log('⚠️  硬盘序列号: 未获取到');
  }
} catch (e) {
  console.log(`⚠️  硬盘序列号: 获取失败`);
}

console.log('');

// ========================================
// 3. 生成新的稳定机器码
// ========================================
console.log('【步骤 3】生成稳定机器码\n');

const components = [macAddress, motherboardSerial, cpuId].filter(c => c);
const combined = components.join('|');
const newMachineId = crypto.createHash('sha256').update(combined).digest('hex');

console.log(`组合字符串: ${combined}`);
console.log(`新机器码 (SHA-256): ${newMachineId}`);
console.log('');

// ========================================
// 4. 对比旧的机器码获取方式
// ========================================
console.log('【步骤 4】对比旧的机器码获取方式（node-machine-id）\n');

let oldMachineId = 'N/A';
try {
  const { machineIdSync } = require('node-machine-id');
  const rawMachineId = machineIdSync({ original: true });
  oldMachineId = crypto.createHash('sha256').update(rawMachineId).digest('hex');
  console.log(`旧机器码 (node-machine-id): ${oldMachineId}`);
  console.log(`原始值: ${rawMachineId}`);
} catch (error) {
  console.log(`❌ 旧方式失败: ${error.message}`);
}

console.log('');

// ========================================
// 5. 对比结果
// ========================================
console.log('【步骤 5】对比分析\n');

if (oldMachineId !== 'N/A' && oldMachineId === newMachineId) {
  console.log('✅ 新旧机器码一致（当前环境下两种方式结果相同）');
} else {
  console.log('⚠️  新旧机器码不同：');
  console.log(`   - 旧机器码基于注册表 MachineGuid，会被 cursor-free-vip 重置`);
  console.log(`   - 新机器码基于物理MAC地址，不受重置影响 ✅`);
}

console.log('');

// ========================================
// 6. 查看缓存
// ========================================
console.log('【步骤 6】机器码缓存信息\n');

const CACHE_DIR = path.join(process.env.APPDATA || os.homedir(), '.cursor-renewal-cache');
const CACHE_FILE = path.join(CACHE_DIR, 'stable-machine-id.json');

console.log(`缓存文件路径: ${CACHE_FILE}`);

if (fs.existsSync(CACHE_FILE)) {
  try {
    const cachedData = JSON.parse(fs.readFileSync(CACHE_FILE, 'utf8'));
    console.log('缓存内容:');
    console.log(JSON.stringify(cachedData, null, 2));
    
    if (cachedData.machineId === newMachineId) {
      console.log('\n✅ 缓存的机器码与当前计算的机器码一致');
    } else {
      console.log('\n⚠️  缓存的机器码与当前计算的机器码不一致');
      console.log('   可能原因: 硬件配置发生了变化');
    }
  } catch (e) {
    console.log(`❌ 读取缓存失败: ${e.message}`);
  }
} else {
  console.log('ℹ️  缓存文件不存在（首次运行会自动创建）');
  
  // 创建缓存
  try {
    if (!fs.existsSync(CACHE_DIR)) {
      fs.mkdirSync(CACHE_DIR, { recursive: true });
    }
    
    const cacheData = {
      machineId: newMachineId,
      components: {
        macAddress,
        motherboardSerial,
        cpuId
      },
      timestamp: new Date().toISOString()
    };
    
    fs.writeFileSync(CACHE_FILE, JSON.stringify(cacheData, null, 2), 'utf8');
    console.log('✅ 已创建缓存文件');
  } catch (e) {
    console.log(`❌ 创建缓存失败: ${e.message}`);
  }
}

console.log('');

// ========================================
// 7. 总结
// ========================================
console.log('='.repeat(80));
console.log('测试总结');
console.log('='.repeat(80));
console.log('');
console.log('✅ 新的机器码获取方式基于以下硬件信息（按优先级）:');
console.log(`   1. 物理MAC地址: ${macAddress} [必需]`);
if (motherboardSerial) {
  console.log(`   2. 主板序列号: ${motherboardSerial} [可选]`);
}
if (cpuId) {
  console.log(`   3. CPU ID: ${cpuId} [可选]`);
}
console.log('');
console.log('📌 重要特性:');
console.log('   • 物理MAC地址是硬件级别的，不会因软件操作而改变');
console.log('   • 即使使用 cursor-free-vip 重置机器码，物理MAC也不会变');
console.log('   • 机器码会被缓存，确保多次获取结果一致');
console.log('   • 如果硬件配置真的变更（如更换网卡），机器码才会改变');
console.log('');
console.log('🔐 最终机器码:');
console.log(`   ${newMachineId}`);
console.log('');
console.log('='.repeat(80));
console.log('');

