import tkinter as tk
import os
import json
import time
from tkinter import ttk, messagebox, filedialog
from pathlib import Path
import hashlib
import shutil
import threading

from build_info import BUILD_TIME

class SyncConfigWindow(tk.Toplevel):
    def __init__(self, master, data, on_update):
        super().__init__(master)
        self.title("同步规则配置")
        self.geometry("800x400")

        self.data = data
        self.on_update = on_update

        columns = ("source", "target")
        self.tree = ttk.Treeview(self, columns=columns, show="headings")
        self.tree.heading("source", text="源文件夹")
        self.tree.heading("target", text="目标文件夹")
        self.tree.pack(fill=tk.BOTH, expand=True, padx=10, pady=10)

        for row in self.data:
            self.tree.insert("", tk.END, values=row)

        input_frame = tk.Frame(self)
        input_frame.pack(fill=tk.X, padx=10, pady=5)

        # ===== 源文件夹 =====
        tk.Label(input_frame, text="源：").grid(row=0, column=0, sticky="e")
        self.src_entry = tk.Entry(input_frame, width=40, state="readonly")
        self.src_entry.grid(row=0, column=1, padx=5)

        tk.Button(
            input_frame,
            text="选择源文件夹",
            command=self.choose_src
        ).grid(row=0, column=2, padx=5)

        # ===== 目标文件夹 =====
        tk.Label(input_frame, text="目标：").grid(row=1, column=0, sticky="e")
        self.dst_entry = tk.Entry(input_frame, width=40, state="readonly")
        self.dst_entry.grid(row=1, column=1, padx=5)

        tk.Button(
            input_frame,
            text="选择目标文件夹",
            command=self.choose_dst
        ).grid(row=1, column=2, padx=5)

        btn_frame = tk.Frame(self)
        btn_frame.pack(pady=8)

        tk.Button(btn_frame, text="添加", command=self.add_row).pack(side=tk.LEFT, padx=5)
        tk.Button(btn_frame, text="删除选中", command=self.delete_row).pack(side=tk.LEFT, padx=5)
        tk.Button(btn_frame, text="确定", command=self.confirm).pack(side=tk.RIGHT, padx=5)

    # ===== 文件夹选择 =====
    def choose_src(self):
        path = filedialog.askdirectory(title="选择源文件夹")
        if path:
            self._set_entry(self.src_entry, path)

    def choose_dst(self):
        path = filedialog.askdirectory(title="选择目标文件夹")
        if path:
            self._set_entry(self.dst_entry, path)

    def _set_entry(self, entry, value):
        entry.config(state=tk.NORMAL)
        entry.delete(0, tk.END)
        entry.insert(0, value)
        entry.config(state="readonly")

    # ===== 规则操作 =====
    def add_row(self):
        src = self.src_entry.get().strip()
        dst = self.dst_entry.get().strip()

        if not src or not dst:
            messagebox.showwarning("提示", "请先选择源和目标文件夹")
            return

        def is_sub_path(parent, child):
            try:
                child_path = Path(child).resolve()
                parent_path = Path(parent).resolve()
                child_path.relative_to(parent_path)
                return True
            except ValueError:
                return False
        
        if is_sub_path(src, dst) or is_sub_path(dst, src):
            messagebox.showerror("错误", "源文件夹和目标文件夹不能是包含关系！")
            return

        self.tree.insert("", tk.END, values=(src, dst))
        self._set_entry(self.src_entry, "")
        self._set_entry(self.dst_entry, "")

    def delete_row(self):
        for item in self.tree.selection():
            self.tree.delete(item)

    def confirm(self):
        self.data.clear()
        for item in self.tree.get_children():
            self.data.append(self.tree.item(item)["values"])
        
        with open("config.cfg", "w", encoding="utf-8") as f:
            json.dump(self.data, f, ensure_ascii=False, indent=2)
        
        self.on_update()
        self.destroy()

class SyncExecuteWindow(tk.Toplevel):
    def __init__(self, master, scan_result):
        super().__init__(master)
        self.title("执行同步")
        self.center_to_parent(master, 900, 420)
        self.resizable(True, True)

        self.scan_result = scan_result

        table_frame = tk.Frame(self)
        table_frame.pack(fill=tk.BOTH, expand=True, padx=10, pady=10)

        columns = ("path", "action", "status")
        self.tree = ttk.Treeview(
            table_frame,
            columns=columns,
            show="headings"
        )

        self.tree.heading("path", text="文件路径")
        self.tree.heading("action", text="操作")
        self.tree.heading("status", text="状态")

        self.tree.column("path", anchor="w", width=600)
        self.tree.column("action", anchor="center", width=100)
        self.tree.column("status", anchor="center", width=120)

        self.tree.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)

        scrollbar = ttk.Scrollbar(
            table_frame,
            orient="vertical",
            command=self.tree.yview
        )
        scrollbar.pack(side=tk.RIGHT, fill=tk.Y)
        self.tree.configure(yscrollcommand=scrollbar.set)

        self.tree.bind(
            "<MouseWheel>",
            lambda e: self.tree.yview_scroll(int(-1 * (e.delta / 120)), "units")
        )

        self.tree.tag_configure("成功", foreground="green")
        self.tree.tag_configure("失败", foreground="red")
        self.tree.tag_configure("待执行", foreground="gray")

        for src, dst, action in self.scan_result:
            display_path = dst if dst else src
            self.tree.insert(
                "",
                tk.END,
                values=(display_path, action, "待执行"),
                tags=("待执行",)
            )

        btn_frame = tk.Frame(self)
        btn_frame.pack(pady=8)

        self.btn_execute = ttk.Button(
            btn_frame,
            text="开始执行",
            width=14,
            command=self.execute_sync
        )
        self.btn_execute.pack(side=tk.LEFT, padx=10)

        self.btn_close = ttk.Button(
            btn_frame,
            text="关闭",
            width=14,
            command=self.destroy
        )
        self.btn_close.pack(side=tk.LEFT, padx=10)

        self.transient(master)
        self.grab_set()
    
    def execute_sync(self):
        self.btn_execute.config(state="disabled")
        self.btn_close.config(state="disabled")

        t = threading.Thread(
            target=self._execute_sync,
            daemon=True
        )
        t.start()

    def _execute_sync(self):
        for index, (src, dst, action) in enumerate(self.scan_result):
            try:
                self._update_status(index, "0%")

                if action == "新增" or action == "覆盖":
                    self._copy_with_progress(src, dst, index)

                elif action == "删除":
                    os.remove(dst)

                self._update_status(index, "成功")

            except Exception as e:
                self._update_status(index, "失败")
                messagebox.showerror(f"同步失败: {e}")

        messagebox.showinfo("完成", "同步执行完成")

    def _copy_with_progress(self, src, dst, index):
        total_size = os.path.getsize(src)
        copied = 0
        chunk_size = 1024 * 1024

        tmp_dst = dst + ".tmp"
        os.makedirs(os.path.dirname(dst), exist_ok=True)

        with open(src, "rb") as fsrc, open(tmp_dst, "wb") as fdst:
            while True:
                chunk = fsrc.read(chunk_size)
                if not chunk:
                    break

                fdst.write(chunk)
                copied += len(chunk)

                percent = int(copied * 100 / total_size)
                self._update_status(index, f"{percent}%")

        shutil.copystat(src, tmp_dst) # 元数据复制

        os.replace(tmp_dst, dst) # 替换

    def _update_status(self, index, status):
        item_id = self.tree.get_children()[index]
        values = list(self.tree.item(item_id, "values"))
        values[2] = status
        self.tree.item(item_id, values=values, tags=(status,))
        self.tree.update_idletasks()

    def center_to_parent(self, parent, width, height):
        parent.update_idletasks()
        px = parent.winfo_x()
        py = parent.winfo_y()
        pw = parent.winfo_width()
        ph = parent.winfo_height()

        x = px + (pw - width) // 2
        y = py + (ph - height) // 2
        self.geometry(f"{width}x{height}+{x}+{y}")

class SyncPreviewWindow(tk.Toplevel):
    def __init__(self, master, data):
        super().__init__(master)
        self.title("同步变更预览")
        self.center_to_parent(master, 800, 400)
        self.resizable(True, True)

        self.data = data

        # ===== 表格 =====
        columns = ("path", "action")
        self.tree = ttk.Treeview(
            self,
            columns=columns,
            show="headings",
            height=15
        )

        self.tree.heading("path", text="文件路径")
        self.tree.heading("action", text="操作")

        self.tree.column("path", anchor="w", width=600)
        self.tree.column("action", anchor="center", width=120)

        self.tree.pack(fill=tk.BOTH, expand=True, padx=10, pady=10)

        # ===== 滚动条 =====
        scrollbar = ttk.Scrollbar(self, orient="vertical", command=self.tree.yview)
        self.tree.configure(yscrollcommand=scrollbar.set)
        scrollbar.place(relx=1.0, rely=0.02, relheight=0.85, anchor="ne")

        # ===== 样式（颜色区分）=====
        self.tree.tag_configure("新增", foreground="green")
        self.tree.tag_configure("覆盖", foreground="orange")
        self.tree.tag_configure("删除", foreground="red")

        # ===== 填充数据 =====
        for src, dst, action in self.data:
            self.tree.insert(
                "",
                tk.END,
                values=(src, action),
                tags=(action,)
            )

        # ===== 底部按钮 =====
        btn_frame = tk.Frame(self)
        btn_frame.pack(pady=8)

        ttk.Button(btn_frame, text="确认", width=12, command=self.confirm).pack(
            side=tk.LEFT, padx=10
        )
        ttk.Button(btn_frame, text="取消", width=12, command=self.cancel).pack(
            side=tk.LEFT, padx=10
        )

        self.grab_set()  # 模态窗口
        self.transient(master)

    def confirm(self):
        self.destroy()
        SyncExecuteWindow(self.master, self.data)

    def cancel(self):
        self.destroy()
    
    def center_to_parent(self, parent, width, height):
        parent.update_idletasks()

        px = parent.winfo_x()
        py = parent.winfo_y()
        pw = parent.winfo_width()
        ph = parent.winfo_height()

        x = px + (pw - width) // 2
        y = py + (ph - height) // 2

        self.geometry(f"{width}x{height}+{x}+{y}")

class ScanProgressWindow(tk.Toplevel):
    def __init__(self, master, total_groups):
        super().__init__(master)
        self.title("扫描进度")
        self.resizable(False, False)

        self.center_to_parent(master, 480, 180)

        self.label = tk.Label(self, text="准备扫描...", anchor="w")
        self.label.pack(fill=tk.X, padx=15, pady=(10, 5))

        tk.Label(self, text="文件夹进度").pack(anchor="w", padx=15)
        self.group_bar = ttk.Progressbar(
            self,
            maximum=total_groups,
            length=440
        )
        self.group_bar.pack(padx=15, pady=(0, 10))

        tk.Label(self, text="文件扫描进度").pack(anchor="w", padx=15)
        self.file_bar = ttk.Progressbar(
            self,
            maximum=1,
            length=440
        )
        self.file_bar.pack(padx=15)

        self.transient(master)
        self.grab_set()

    def center_to_parent(self, parent, width, height):
        self.update_idletasks()
        parent.update_idletasks()

        px = parent.winfo_x()
        py = parent.winfo_y()
        pw = parent.winfo_width()
        ph = parent.winfo_height()

        x = px + (pw - width) // 2
        y = py + (ph - height) // 2

        self.geometry(f"{width}x{height}+{x}+{y}")

    def update_group_progress(self, value, text=None):
        self.group_bar["value"] = value
        if text:
            self.label.config(text=text)
        self.update_idletasks()

    def reset_file_progress(self, total_files):
        self.file_bar["maximum"] = max(total_files, 1)
        self.file_bar["value"] = 0
        self.update_idletasks()

    def update_file_progress(self, value):
        self.file_bar["value"] = value
        self.update_idletasks()

class SyncApp(tk.Tk):
    def __init__(self):
        super().__init__()
        self.title("文件夹同步工具")
        self.center_window(650, 520)

        tk.Label(self, text="日志：").pack(anchor="w", padx=10)

        log_frame = tk.Frame(self)
        log_frame.pack(fill=tk.BOTH, padx=10, pady=5)

        scrollbar = tk.Scrollbar(log_frame)
        scrollbar.pack(side=tk.RIGHT, fill=tk.Y)

        self.text_box = tk.Text(
            log_frame,
            height=6,
            state=tk.DISABLED,
            wrap=tk.WORD,
            yscrollcommand=scrollbar.set
        )
        self.text_box.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        scrollbar.config(command=self.text_box.yview)

        self.text_box.tag_config("info", foreground="black")
        self.text_box.tag_config("success", foreground="green")
        self.text_box.tag_config("error", foreground="red")

        self.log(f"Folder Sync 工具 1.0 (打包时间：{BUILD_TIME}, 作者: Huang)")

        btn_frame = tk.Frame(self)
        btn_frame.pack(pady=8)

        tk.Button(btn_frame, text="开始同步", width=14, command=self.start_sync).pack(side=tk.LEFT, padx=10)
        tk.Button(btn_frame, text="配置同步规则", width=14, command=self.open_config_window).pack(side=tk.LEFT, padx=10)

        if os.path.exists("config.cfg"):
            self.log("加载同步规则配置..." , end="")
            try:
                with open("config.cfg", "r", encoding="utf-8") as f:
                    self.sync_rules = json.load(f)
                self.log("成功。", ptime = False)
            except Exception as e:
                self.log_error(f"失败！", ptime = False)
                self.log_error(f"读取同步规则配置时出错：{e}，将使用空配置。")
                self.sync_rules = []
        else:
            self.sync_rules = []
            self.log("未找到同步规则配置，当前为空。")

        tk.Label(self, text="同步配置：").pack(anchor="w", padx=10)

        self.listbox = tk.Listbox(self)
        self.listbox.pack(fill=tk.BOTH, expand=True, padx=10, pady=5)

        self.refresh_list()
        self.log_success("已完成初始化。")

    def log(self, msg, end="\n", ptime=True):
        now = time.strftime("%Y-%m-%d %H:%M", time.localtime())
        if ptime:
            self._append_log(f"[{now}] {msg}", "info", end)
        else:
            self._append_log(f"{msg}", "info", end)

    def log_success(self, msg, end="\n", ptime=True):
        now = time.strftime("%Y-%m-%d %H:%M", time.localtime())
        if ptime:
            self._append_log(f"[{now}] {msg}", "success", end)
        else:
            self._append_log(f"{msg}", "success", end)

    def log_error(self, msg, end="\n", ptime=True):
        now = time.strftime("%Y-%m-%d %H:%M", time.localtime())
        if ptime:
            self._append_log(f"[{now}] {msg}", "error", end)
        else:
            self._append_log(f"{msg}", "error", end)

    def _append_log(self, msg, tag, end="\n"):
        self.text_box.config(state=tk.NORMAL)
        self.text_box.insert(tk.END, f"{msg} {end}", tag)
        self.text_box.see(tk.END)
        self.text_box.config(state=tk.DISABLED)

    # 业务逻辑
    def center_window(self, width, height):
        self.update_idletasks()

        screen_width = self.winfo_screenwidth()
        screen_height = self.winfo_screenheight()

        x = (screen_width - width) // 2
        y = (screen_height - height) // 2

        self.geometry(f"{width}x{height}+{x}+{y}")

    def open_config_window(self):
        SyncConfigWindow(self, self.sync_rules, self.refresh_list)

    def refresh_list(self):
        self.listbox.delete(0, tk.END)
        for src, dst in self.sync_rules:
            self.listbox.insert(tk.END, f"{src}  →  {dst}")

    def gethash(self, filename, hash_factory=hashlib.sha256, chunk_num_blocks=128):
        h = hash_factory()
        with open(filename,'rb') as f: 
            while chunk := f.read(chunk_num_blocks*h.block_size): 
                h.update(chunk)
        return h.hexdigest()

    def start_sync(self):
        if not self.sync_rules:
            messagebox.showwarning("提示", "尚未配置任何同步规则")
            return

        scan_result = []

        progress_win = ScanProgressWindow(
            self,
            total_groups=len(self.sync_rules)
        )

        self.log("开始扫描文件夹..." , end="")

        """
        文件同步有且只有三种可能，对于A -> B来说：
        1. 新增：A有，B没有
        2. 覆盖：A有，B有，但内容不同
        3. 删除：A没有，B有
        """

        for group_index, (src_dir, dst_dir) in enumerate(self.sync_rules, start=1):

            progress_win.update_group_progress(
                group_index,
                text=f"扫描文件夹：{src_dir} → {dst_dir}"
            )

            if not os.path.exists(dst_dir):
                messagebox.showerror("错误", f"目标文件夹不存在：{dst_dir}")
                self.log_error(f"错误。",ptime=False)
                self.log_error(f"目标文件夹不存在：{dst_dir}，同步终止。")
                return

            source_files = []
            for root, _, files in os.walk(src_dir):
                for filename in files:
                    full_path = os.path.join(root, filename)
                    source_files.append(full_path)

            # 重置文件进度条
            progress_win.reset_file_progress(len(source_files))

            # 检查新增和覆盖的情况
            for file_index, src_file in enumerate(source_files, start=1):
                progress_win.update_group_progress(
                    group_index,
                    text=f"扫描文件：{src_file}"
                )

                # 计算目标文件路径
                relative_path = os.path.relpath(src_file, src_dir) # 相对路径
                dst_file = os.path.join(dst_dir, relative_path) # 目标文件夹的这个文件的路径

                if not os.path.exists(dst_file): # 目标文件夹不存在这个文件
                    scan_result.append((src_file, dst_file, "新增"))
                else:
                    # 目标文件夹存在这个文件，比较md5
                    src_sha256 = self.gethash(src_file)
                    dst_sha256 = self.gethash(dst_file)

                    if src_sha256 != dst_sha256:
                        scan_result.append((src_file, dst_file, "覆盖"))

                # 更新文件进度条
                progress_win.update_file_progress(file_index)

            # 检查删除的情况
            for root, _, files in os.walk(dst_dir):
                for filename in files:
                    dst_file = os.path.join(root, filename)
                    relative_path = os.path.relpath(dst_file, dst_dir)
                    src_file = os.path.join(src_dir, relative_path)

                    if not os.path.exists(src_file):
                        scan_result.append((src_file, dst_file, "删除"))

        progress_win.destroy()
        self.log("完成。" , ptime=False)

        SyncPreviewWindow(self, scan_result)

if __name__ == "__main__":
    app = SyncApp()
    app.iconbitmap("icon.ico")
    app.mainloop()
